<?php
header('Content-Type: application/json');

// Configuración de la base de datos
$db_path = __DIR__.'/../api/.db.db';

try {
    $db = new SQLite3($db_path);
    $db->busyTimeout(5000);
    $db->exec("PRAGMA journal_mode = WAL");
} catch (Exception $e) {
    die(json_encode(['error' => "Error al conectar con la base de datos: " . $e->getMessage()]));
}

// Obtener todos los DNS
$dnss = $db->query("SELECT * FROM dns");
$dnsMap = [];
while ($dns = $dnss->fetchArray(SQLITE3_ASSOC)) {
    $dnsMap[$dns['id']] = $dns['title'];
}

// Función para formatear MAC
function formatMac($mac) {
    $mac = strtoupper(preg_replace('/[^0-9A-F]/', '', $mac));
    return implode(':', str_split($mac, 2));
}

// Determinar la acción solicitada
$action = $_GET['action'] ?? $_POST['action'] ?? 'search';

try {
    switch ($action) {
        case 'search':
            $searchTerm = $_GET['search'] ?? '';
            $results = [];
            
            if (!empty($searchTerm)) {
                $stmt = $db->prepare("SELECT * FROM playlist WHERE mac_address LIKE :mac OR cliente_nombre LIKE :name");
                $stmt->bindValue(':mac', '%'.$searchTerm.'%', SQLITE3_TEXT);
                $stmt->bindValue(':name', '%'.$searchTerm.'%', SQLITE3_TEXT);
                $result = $stmt->execute();
                
                while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
                    $results[] = $row;
                }
            }
            
            echo json_encode([
                'success' => true,
                'results' => $results,
                'dnsMap' => $dnsMap
            ]);
            break;
            
        case 'get_client':
            $id = intval($_GET['id']);
            
            $stmt = $db->prepare("SELECT * FROM playlist WHERE id = :id");
            $stmt->bindValue(':id', $id, SQLITE3_INTEGER);
            $result = $stmt->execute();
            $client = $result->fetchArray(SQLITE3_ASSOC);
            
            if ($client) {
    // ✅ Evita que el navegador reste un día por zona horaria
    $client['expiracion'] = date('Y-m-d', strtotime($client['expiracion']));

    echo json_encode([
        'success' => true,
        'client' => $client,
        'dnsMap' => $dnsMap
    ]);
            } else {
                throw new Exception("Cliente no encontrado");
            }
            break;
            
        case 'toggle_protect':
            $id = intval($_GET['id']);
            
            // Obtener estado actual
            $stmt = $db->prepare("SELECT is_protected FROM playlist WHERE id = :id");
            $stmt->bindValue(':id', $id, SQLITE3_INTEGER);
            $result = $stmt->execute();
            $current = $result->fetchArray(SQLITE3_ASSOC);
            
            if ($current) {
                $new_value = $current['is_protected'] ? 0 : 1;
                
                // Actualizar estado
                $stmt = $db->prepare("UPDATE playlist SET is_protected = :protect WHERE id = :id");
                $stmt->bindValue(':protect', $new_value, SQLITE3_INTEGER);
                $stmt->bindValue(':id', $id, SQLITE3_INTEGER);
                
                if ($stmt->execute()) {
                    echo json_encode([
                        'success' => true,
                        'new_status' => $new_value,
                        'message' => "Protección " . ($new_value ? "activada" : "desactivada")
                    ]);
                } else {
                    throw new Exception("Error al cambiar el estado de protección");
                }
            } else {
                throw new Exception("Cliente no encontrado");
            }
            break;
            
        case 'update_client':
    $id = intval($_POST['id']);
    
    // Validar y formatear la fecha correctamente
    $expiracion = $_POST['expiracion'];
    if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $expiracion)) {
        throw new Exception("Formato de fecha inválido");
    }

    // Crear objeto DateTime sin considerar la zona horaria
    $fechaObj = DateTime::createFromFormat('Y-m-d', $expiracion, new DateTimeZone('UTC'));
    $fechaObj->setTime(12, 0, 0); // Establecer hora del día para evitar problemas
            
            // Preparar datos
            $data = [
                'mac_address' => formatMac($_POST['mac_address']),
                'username' => trim($_POST['username']),
                'password' => trim($_POST['password']),
                'dns_id' => intval($_POST['dns_id']),
                'cliente_nombre' => trim($_POST['cliente_nombre']),
                'cliente_whatsapp' => trim($_POST['cliente_whatsapp']),
                'expiracion' => $fechaObj->format('Y-m-d'),
                'is_protected' => isset($_POST['is_protected']) ? 1 : 0
            ];
            
           // Validación de número de WhatsApp (con o sin +)
if (!preg_match('/^\+?\d{7,15}$/', $data['cliente_whatsapp'])) {
    throw new Exception("Formato de WhatsApp  (ej: 593980649398)");
}

        
            if (!preg_match('/^([0-9A-Fa-f]{2}[:-]){2,7}[0-9A-Fa-f]{2}$/', $data['mac_address'])) {
                throw new Exception("Formato de MAC inválido. Debe tener entre 3 y 8 pares hexadecimales");
            }
            
            if (strtotime($data['expiracion']) < strtotime(date('Y-m-d'))) {
                throw new Exception("La fecha de vencimiento no puede ser en el pasado");
            }
            
            // Actualizar en base de datos
            $stmt = $db->prepare("UPDATE playlist SET 
                mac_address = :mac_address,
                username = :username,
                password = :password,
                dns_id = :dns_id,
                cliente_nombre = :cliente_nombre,
                cliente_whatsapp = :cliente_whatsapp,
                expiracion = :expiracion,
                is_protected = :is_protected
                WHERE id = :id");
            
            foreach ($data as $key => $value) {
                $stmt->bindValue(":$key", $value, is_int($value) ? SQLITE3_INTEGER : SQLITE3_TEXT);
            }
            $stmt->bindValue(':id', $id, SQLITE3_INTEGER);
            
            if ($stmt->execute()) {
                echo json_encode([
                    'success' => true,
                    'message' => "Cliente actualizado correctamente"
                ]);
            } else {
                throw new Exception("Error al actualizar el cliente");
            }
            break;
            
        case 'delete_client':
            $id = intval($_POST['id']);
            
            $stmt = $db->prepare("DELETE FROM playlist WHERE id = :id");
            $stmt->bindValue(':id', $id, SQLITE3_INTEGER);
            
            if ($stmt->execute()) {
                echo json_encode([
                    'success' => true,
                    'message' => "Cliente eliminado correctamente"
                ]);
            } else {
                throw new Exception("Error al eliminar el cliente");
            }
            break;
            
        default:
            throw new Exception("Acción no válida");
    }
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

$db->close();
?>